<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Profile;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProfileController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $profiles = Profile::ordered()->paginate(15);
        return view('admin.profiles.index', compact('profiles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.profiles.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'category' => 'nullable|string|max:100',
            'excerpt' => 'nullable|string|max:500',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'order' => 'nullable|integer',
            'is_active' => 'nullable|boolean',
        ]);

        $data = $request->all();

        // Handle checkbox is_active - if not present in request, it means unchecked
        $data['is_active'] = $request->has('is_active');

        // Generate slug from title
        $data['slug'] = Str::slug($request->title);

        // Ensure unique slug
        $originalSlug = $data['slug'];
        $count = 1;
        while (Profile::where('slug', $data['slug'])->exists()) {
            $data['slug'] = $originalSlug . '-' . $count;
            $count++;
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $imageName = time() . '.' . $request->image->extension();
            $request->image->move(public_path('images/profiles'), $imageName);
            $data['image'] = 'images/profiles/' . $imageName;
        }

        Profile::create($data);

        return redirect()->route('admin.profiles.index')
            ->with('success', 'Profile berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Profile $profile)
    {
        return view('admin.profiles.show', compact('profile'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Profile $profile)
    {
        return view('admin.profiles.edit', compact('profile'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Profile $profile)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'category' => 'nullable|string|max:100',
            'excerpt' => 'nullable|string|max:500',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'order' => 'nullable|integer',
            'is_active' => 'nullable|boolean',
        ]);

        $data = $request->all();

        // Handle checkbox is_active - if not present in request, it means unchecked
        $data['is_active'] = $request->has('is_active');

        // Generate slug from title if title changed
        if ($request->title !== $profile->title) {
            $data['slug'] = Str::slug($request->title);

            // Ensure unique slug
            $originalSlug = $data['slug'];
            $count = 1;
            while (Profile::where('slug', $data['slug'])->where('id', '!=', $profile->id)->exists()) {
                $data['slug'] = $originalSlug . '-' . $count;
                $count++;
            }
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image
            if ($profile->image && file_exists(public_path($profile->image))) {
                unlink(public_path($profile->image));
            }

            $imageName = time() . '.' . $request->image->extension();
            $request->image->move(public_path('images/profiles'), $imageName);
            $data['image'] = 'images/profiles/' . $imageName;
        }

        $profile->update($data);

        return redirect()->route('admin.profiles.index')
            ->with('success', 'Profile berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Profile $profile)
    {
        // Delete image file
        if ($profile->image && file_exists(public_path($profile->image))) {
            unlink(public_path($profile->image));
        }

        $profile->delete();

        return redirect()->route('admin.profiles.index')
            ->with('success', 'Profile berhasil dihapus.');
    }
}
