<?php

namespace App\Http\Controllers;

use App\Models\Document;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class DocumentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $documents = Document::orderBy('created_at', 'desc')->paginate(15);

        return view('admin.documents.index', compact('documents'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.documents.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'file' => 'required|file|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx|max:20480', // 20MB max
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // 2MB max untuk preview image
            'category' => 'required|string|max:100',
            'is_active' => 'boolean'
        ]);

        $file = $request->file('file');
        $fileName = time() . '_' . Str::slug($file->getClientOriginalName()) . '.' . $file->getClientOriginalExtension();
        $filePath = $file->storeAs('documents', $fileName, 'public');

        $imagePath = null;
        if ($request->hasFile('image')) {
            $imageFile = $request->file('image');
            $imageName = time() . '_preview_' . Str::slug($imageFile->getClientOriginalName()) . '.' . $imageFile->getClientOriginalExtension();
            $imagePath = $imageFile->storeAs('document_previews', $imageName, 'public');
        }

        Document::create([
            'title' => $request->title,
            'description' => $request->description,
            'file_name' => $file->getClientOriginalName(),
            'file_path' => $filePath,
            'file_size' => $file->getSize(),
            'file_type' => $file->getMimeType(),
            'image' => $imagePath,
            'category' => $request->category,
            'is_active' => $request->has('is_active')
        ]);

        return redirect()->route('admin.documents.index')
            ->with('success', 'Dokumen berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $document = Document::findOrFail($id);

        return view('admin.documents.show', compact('document'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $document = Document::findOrFail($id);

        return view('admin.documents.edit', compact('document'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $document = Document::findOrFail($id);

        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'file' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx|max:20480', // 20MB max
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048', // 2MB max untuk preview image
            'category' => 'required|string|max:100',
            'is_active' => 'boolean'
        ]);

        $updateData = [
            'title' => $request->title,
            'description' => $request->description,
            'category' => $request->category,
            'is_active' => $request->has('is_active')
        ];

        // Jika ada file baru yang diupload
        if ($request->hasFile('file')) {
            // Hapus file lama
            if ($document->file_path && Storage::disk('public')->exists($document->file_path)) {
                Storage::disk('public')->delete($document->file_path);
            }

            $file = $request->file('file');
            $fileName = time() . '_' . Str::slug($file->getClientOriginalName()) . '.' . $file->getClientOriginalExtension();
            $filePath = $file->storeAs('documents', $fileName, 'public');

            $updateData['file_name'] = $file->getClientOriginalName();
            $updateData['file_path'] = $filePath;
            $updateData['file_size'] = $file->getSize();
            $updateData['file_type'] = $file->getMimeType();
        }

        // Jika ada image preview baru yang diupload
        if ($request->hasFile('image')) {
            // Hapus image lama
            if ($document->image && Storage::disk('public')->exists($document->image)) {
                Storage::disk('public')->delete($document->image);
            }

            $imageFile = $request->file('image');
            $imageName = time() . '_preview_' . Str::slug($imageFile->getClientOriginalName()) . '.' . $imageFile->getClientOriginalExtension();
            $imagePath = $imageFile->storeAs('document_previews', $imageName, 'public');

            $updateData['image'] = $imagePath;
        }

        $document->update($updateData);

        return redirect()->route('admin.documents.index')
            ->with('success', 'Dokumen berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $document = Document::findOrFail($id);

        // Hapus file dari storage
        if ($document->file_path && Storage::disk('public')->exists($document->file_path)) {
            Storage::disk('public')->delete($document->file_path);
        }

        // Hapus image preview dari storage
        if ($document->image && Storage::disk('public')->exists($document->image)) {
            Storage::disk('public')->delete($document->image);
        }

        $document->delete();

        return redirect()->route('admin.documents.index')
            ->with('success', 'Dokumen berhasil dihapus.');
    }

    /**
     * Download document file or display preview.
     */
    public function download($id)
    {
        $document = Document::findOrFail($id);

        if (!$document->fileExists()) {
            return redirect()->back()->with('error', 'File tidak ditemukan.');
        }

        // Check if this is a preview request
        if (request()->has('preview')) {
            // Return PDF inline for preview (don't increment download count for previews)
            return response()->file(
                Storage::disk('public')->path($document->file_path),
                [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="' . $document->file_name . '"'
                ]
            );
        }

        // Increment download count for actual downloads
        $document->incrementDownloadCount();

        return Storage::disk('public')->download(
            $document->file_path,
            $document->file_name
        );
    }
}
