<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;

class Document extends Model
{
    protected $fillable = [
        'title',
        'description',
        'file_name',
        'file_path',
        'file_size',
        'file_type',
        'image',
        'category',
        'download_count',
        'is_active'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'download_count' => 'integer',
        'file_size' => 'integer'
    ];

    // Scope untuk dokumen aktif
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    // Scope berdasarkan kategori
    public function scopeByCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    // Method untuk format ukuran file
    public function getFormattedFileSizeAttribute()
    {
        $bytes = $this->file_size;
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }

    // Method untuk mendapatkan URL download
    public function getDownloadUrlAttribute()
    {
        return route('documents.download', $this->id);
    }

    // Method untuk increment download count
    public function incrementDownloadCount()
    {
        $this->increment('download_count');
    }

    // Method untuk check apakah file ada
    public function fileExists()
    {
        return Storage::disk('public')->exists($this->file_path);
    }

    // Method untuk mendapatkan path lengkap file
    public function getFullFilePathAttribute()
    {
        return storage_path('app/public/' . $this->file_path);
    }
}
