<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MaintenanceRequest;
use App\Models\Barang;
use App\Models\Laboratorium;
use App\Models\Prodi;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{
    /**
     * FUNGSI HELPER: Filter maintenance otomatis berdasarkan Role.
     */
    private function getMaintenanceData()
    {
        $user = Auth::user();
        $query = MaintenanceRequest::with(['barang.lab', 'user', 'lab']);

        if ($user->role == 'Kaprodi') {
            $query->whereHas('lab', function($q) use ($user) {
                $q->where('prodi_id', $user->prodi_id);
            });
        }
        
        if ($user->role == 'Kepala Lab') {
            $query->where('id_lab', $user->lab_id);
        }

        return $query->latest()->paginate(10);
    }

    /**
     * FUNGSI HELPER: Ambil data semua aset dengan filter (Shared for Pudir 1 & 2).
     */
    private function getTotalAssetsData(Request $request)
    {
        $query = Barang::with(['lab.prodi']);

        // Filter berdasarkan Prodi
        if ($request->filled('prodi_id')) {
            $query->whereHas('lab', function($q) use ($request) {
                $q->where('prodi_id', $request->prodi_id);
            });
        }

        // Filter berdasarkan Kondisi
        if ($request->filled('status_kondisi')) {
            $query->where('status_kondisi', $request->status_kondisi);
        }

        // Filter Pencarian (Nama Alat atau Kode BMN)
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_barang', 'like', "%{$search}%")
                  ->orWhere('kode_bmn', 'like', "%{$search}%");
            });
        }

        return $query->latest()->paginate(20)->withQueryString();
    }

    public function index()
    {
        $maintenances = $this->getMaintenanceData();
        $barangs = Barang::all(); 
        $labs = Laboratorium::all(); 
        return view('dashboard', compact('maintenances', 'barangs', 'labs'));
    }

    /**
     * Role-based redirectors (Standard PHP Method Syntax)
     */
    public function SuperAdmin()
    {
        return $this->index();
    }

    public function KaLab()
    {
        $user = Auth::user();
        $maintenances = $this->getMaintenanceData();
        $barangs = Barang::where('id_lab', $user->lab_id)->get();
        $labs = Laboratorium::where('id', $user->lab_id)->get(); 
        return view('dashboard', compact('maintenances', 'barangs', 'labs'));
    }

    public function TimPemelihara()
    {
        return $this->index();
    }

    /**
     * DASHBOARD PEMBANTU DIREKTUR 1 (BIDANG AKADEMIK)
     */
    public function PembantuDirektur1(Request $request)
    {
        $menu = $request->route()->defaults['menu'] ?? 'index';

        switch ($menu) {
            case 'index':
                $approvalQueue = MaintenanceRequest::with(['barang.lab', 'user', 'lab'])
                                    ->where('status', 'pending_pudir1')
                                    ->latest()->paginate(10);
                return view('dashboard', [
                    'maintenances' => $approvalQueue,
                    'barangs' => Barang::all(),
                    'labs' => Laboratorium::all(),
                    'title' => 'Antrean Verifikasi Akademik'
                ]);
            
            case 'readiness':
                $readiness = Laboratorium::withCount([
                    'barangs as total_alat',
                    'barangs as alat_siap' => function($q) { $q->where('status_kondisi', 'Normal'); }
                ])->get();
                return view('admin.pudir1.readiness', compact('readiness'));
            
            case 'high_impact':
                $highImpact = MaintenanceRequest::with(['barang.lab', 'user', 'lab'])
                    ->whereHas('barang', function($q) {
                        $q->where('kategori', 'Alat Praktikum');
                    })
                    ->whereIn('status', ['repairing', 'checking_technical', 'pending_pudir1'])
                    ->paginate(10);
                return view('admin.pudir1.high_impact', compact('highImpact'));
            
            case 'calibration':
                $terkalibrasi = Barang::where('next_calibration', '>', now())->count();
                $expired = Barang::where('next_calibration', '<=', now())->count();
                $upcoming = Barang::with('lab')->whereBetween('next_calibration', [now(), now()->addDays(30)])->get();
                return view('admin.pudir1.calibration', compact('terkalibrasi', 'expired', 'upcoming'));
            
            case 'downtime':
                $downtime = MaintenanceRequest::with(['barang', 'lab'])->where('status', 'repairing')->orderBy('updated_at', 'asc')->paginate(10);
                return view('admin.pudir1.downtime', compact('downtime'));

            case 'total_assets':
                $assets = $this->getTotalAssetsData($request);
                $prodis = Prodi::all();
                return view('admin.pudir_shared.total_assets', compact('assets', 'prodis'));

            default:
                return redirect()->route('pudir1.index');
        }
    }

    /**
     * DASHBOARD PEMBANTU DIREKTUR 2 (BIDANG KEUANGAN & ASET)
     */
    public function PembantuDirektur2(Request $request)
    {
        $menu = $request->route()->defaults['menu'] ?? 'index';

        switch ($menu) {
            case 'index':
                $approvalQueue = MaintenanceRequest::where('status', 'pending_pudir2')->latest()->paginate(10);
                $totalRealisasi = MaintenanceRequest::where('status', 'closed')->sum('estimated_cost');
                return view('admin.pudir2.index', compact('approvalQueue', 'totalRealisasi'));

            case 'approval':
                $approvalQueue = MaintenanceRequest::with(['barang.lab', 'user', 'lab'])
                                    ->where('status', 'pending_pudir2')->latest()->paginate(10);
                return view('admin.pudir2.approval', compact('approvalQueue'));

            case 'budget':
                $completedMaintenance = MaintenanceRequest::with(['barang.lab', 'user', 'lab'])
                                            ->whereIn('status', ['repairing', 'waiting_verification', 'closed'])
                                            ->whereNotNull('estimated_cost')->latest('updated_at')->get(); 
                return view('admin.pudir2.budget', compact('completedMaintenance'));

            case 'assets':
                $assetHealth = Barang::with('lab')->where('tahun_perolehan', '<', now()->year - 10)->get();
                return view('admin.pudir2.assets', compact('assetHealth'));
                
            case 'vendor':
                $vendorRepairs = MaintenanceRequest::with(['barang', 'lab'])->where('repair_type', 'External')->latest()->paginate(10);
                return view('admin.pudir2.vendor', compact('vendorRepairs'));

            case 'total_assets':
                $assets = $this->getTotalAssetsData($request);
                $prodis = Prodi::all();
                return view('admin.pudir_shared.total_assets', compact('assets', 'prodis'));

            default:
                return redirect()->route('pudir2.index');
        }
    }
}