<?php

namespace App\Http\Controllers\Kaprodi;

use App\Http\Controllers\Controller;
use App\Models\Laboratorium;
use App\Models\MaintenanceRequest;
use App\Models\Barang;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * DASHBOARD UTAMA KAPRODI
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = MaintenanceRequest::with(['barang', 'lab', 'user'])
            ->whereHas('lab', function($q) use ($user) {
                $q->where('prodi_id', $user->prodi_id);
            });

        if ($request->filled('lab_id')) {
            $query->where('id_lab', $request->lab_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [
                Carbon::parse($request->start_date)->startOfDay(), 
                Carbon::parse($request->end_date)->endOfDay()
            ]);
        }

        $stats = [
            'total'       => (clone $query)->count(),
            'pending'     => (clone $query)->where('status', 'pending_kaprodi')->count(),
            'on_progress' => (clone $query)->whereIn('status', ['repairing', 'checking_technical', 'pending_pudir2'])->count(),
            'closed'      => (clone $query)->where('status', 'closed')->count(),
        ];

        $labs = Laboratorium::where('prodi_id', $user->prodi_id)->orderBy('nama_lab')->get();
        $barangs = Barang::whereHas('lab', function($q) use ($user) {
            $q->where('prodi_id', $user->prodi_id);
        })->orderBy('nama_barang', 'asc')->get();

        $requests = $query->latest()->paginate(10)->withQueryString();

        return view('kaprodi.dashboard', compact('stats', 'labs', 'requests', 'barangs'));
    }

    /**
     * INVENTORY ALAT PER PRODI
     */
    public function inventory()
    {
        $user = Auth::user();
        
        // Menampilkan seluruh barang yang ada di Lab milik Prodi tersebut
        $equipment = Barang::whereHas('lab', function($q) use ($user) {
            $q->where('prodi_id', $user->prodi_id);
        })->with(['lab'])->get();
            
        return view('kaprodi.inventory', compact('equipment'));
    }

    /**
     * MONITORING JADWAL KALIBRASI (FIX ERROR: Undefined Method)
     * Menampilkan alat yang mendekati masa kalibrasi di lingkungan prodi.
     */
    public function calibration()
    {
        $user = Auth::user();
        
        // Kita simpan ke variabel $needs_calibration agar sinkron dengan View
        $needs_calibration = Barang::whereHas('lab', function($q) use ($user) {
                $q->where('prodi_id', $user->prodi_id);
            })
            ->whereNotNull('next_calibration')
            ->orderBy('next_calibration', 'asc')
            ->get();

        // Kirim dengan nama $needs_calibration
        return view('kaprodi.calibration', compact('needs_calibration'));
    }

    /**
     * LAPORAN BIAYA
     */
    public function costReport()
    {
        $user = Auth::user();
        $validStatuses = ['repairing', 'waiting_verification', 'ready_to_close', 'closed'];

        $totalBiaya = MaintenanceRequest::whereHas('lab', function($q) use ($user) {
                $q->where('prodi_id', $user->prodi_id);
            })
            ->whereIn('status', $validStatuses)
            ->sum('estimated_cost');

        $biayaPerLab = MaintenanceRequest::whereIn('maintenance_requests.status', $validStatuses)
            ->join('laboratoriums', 'maintenance_requests.id_lab', '=', 'laboratoriums.id')
            ->where('laboratoriums.prodi_id', $user->prodi_id)
            ->select('laboratoriums.nama_lab as lab_name', DB::raw('SUM(maintenance_requests.estimated_cost) as total'))
            ->groupBy('laboratoriums.nama_lab', 'laboratoriums.id')
            ->get();

        $monthlyTotals = [];
        $monthLabels = [];

        for ($i = 5; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $monthLabels[] = $date->translatedFormat('F');

            $total = MaintenanceRequest::whereHas('lab', function($q) use ($user) {
                    $q->where('prodi_id', $user->prodi_id);
                })
                ->whereIn('status', $validStatuses)
                ->whereMonth('created_at', $date->month)
                ->whereYear('created_at', $date->year)
                ->sum('estimated_cost');

            $monthlyTotals[] = $total;
        }

        return view('kaprodi.cost_report', [
            'totalBiaya' => $totalBiaya,
            'biayaPerLab' => $biayaPerLab,
            'monthlyTotals' => $monthlyTotals,
            'monthLabels' => $monthLabels
        ]);
    }

    /**
     * RIWAYAT SERVIS PER ALAT
     */
    public function equipmentHistory($id)
    {
        $item = Barang::with(['maintenanceRequests' => function($q) {
            $q->with(['lab', 'user'])->latest();
        }])->findOrFail($id);

        return view('kaprodi.equipment_history', compact('item'));
    }
}