<?php

namespace App\Http\Controllers;

use App\Models\MaintenanceRequest;
use App\Models\Barang; 
use App\Models\Laboratorium;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class MaintenanceController extends Controller
{
    /**
     * Menampilkan daftar transaksi perbaikan dengan FILTER OTOMATIS BERDASARKAN ROLE.
     * Mengatasi Bug list kosong dengan menyesuaikan lab_id dan prodi_id user.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $search = $request->input('search');

        // Query dasar menggunakan relasi yang konsisten (barang, lab, user)
        $query = MaintenanceRequest::with(['barang.lab', 'user', 'lab']);

        // 1. FILTERING KEAMANAN DATA (RBAC)
        if ($user->role == 'Kepala Lab') {
            // Filter: Kepala Lab hanya melihat pengajuan dari Lab unitnya sendiri
            $query->where('id_lab', $user->lab_id);
        } elseif ($user->role == 'Kaprodi') {
            // Filter: Kaprodi melihat semua pengajuan dari Lab di bawah prodinya (TPK/TPPK/TPKP)
            $query->whereHas('lab', function($q) use ($user) {
                $q->where('prodi_id', $user->prodi_id);
            });
        }

        // 2. FITUR PENCARIAN (Global Search)
        $query->when($search, function ($q) use ($search) {
            $q->where(function($sub) use ($search) {
                $sub->where('issue_description', 'like', "%{$search}%")
                    ->orWhere('status', 'like', "%{$search}%")
                    ->orWhereHas('barang', function ($b) use ($search) {
                        $b->where('nama_barang', 'like', "%{$search}%")
                          ->orWhere('kode_bmn', 'like', "%{$search}%");
                    });
            });
        });

        $maintenances = $query->latest()->paginate(10)->withQueryString();

        // 3. PENGAMBILAN DATA FORM (Hanya barang yang sesuai unit login)
        $barangs = ($user->role == 'Kepala Lab') 
            ? Barang::where('id_lab', $user->lab_id)->get() 
            : Barang::all();
            
        $labs = Laboratorium::all();

        return view('dashboard', compact('maintenances', 'barangs', 'labs'));
    }

    /**
     * MENYIMPAN PENGAJUAN BARU (STORE)
     * Mengatasi Bug pengajuan tidak muncul dengan mengunci id_lab secara otomatis.
     */
    public function store(Request $request) 
    {
        $user = Auth::user();

        $request->validate([
            'barang_id' => 'required|exists:barangs,id',
            'issue_description' => 'required|string|min:10',
            'damage_level' => 'required|in:Ringan,Sedang,Berat',
            'foto_kerusakan' => 'nullable|image|mimes:jpeg,png,jpg|max:2048'
        ]);

        return DB::transaction(function () use ($request, $user) {
            
            // Logika Upload Foto Kerusakan
            $fotoKerusakan = null;
            if ($request->hasFile('foto_kerusakan')) {
                $fotoKerusakan = $request->file('foto_kerusakan')->store('assets/maintenance/issues', 'public');
            }

            // SIMPAN DATA: id_lab diambil otomatis dari user login
            MaintenanceRequest::create([
                'user_id' => $user->id,
                'barang_id' => $request->barang_id,
                'id_lab' => $user->lab_id, // OTOMATIS: Menghubungkan ke 17 unit kerja yang tepat
                'issue_description' => $request->issue_description,
                'damage_level' => $request->damage_level,
                'status' => 'pending_kaprodi',
                'request_date' => now(),
                'foto_kerusakan' => $fotoKerusakan 
            ]);
            
            return redirect()->route('dashboard')->with('success', 'Pengajuan berhasil dikirim untuk unit ' . $user->lab->nama_lab);
        });
    }

    /**
     * TAHAP 1: Persetujuan KaProdi (TPK/TPPK/TPKP)
     */
    public function approveKaprodi($id) 
    {
        $req = MaintenanceRequest::where('status', 'pending_kaprodi')->findOrFail($id);
        $req->update(['status' => 'pending_pudir1']);
        return redirect()->route('dashboard')->with('success', 'Disetujui Kaprodi. Menunggu Pudir 1.');
    }

    /**
     * TAHAP 2: Persetujuan Pudir 1 (Kesiapan Akademik)
     */
    public function approvePudir1($id) 
    {
        $req = MaintenanceRequest::where('status', 'pending_pudir1')->findOrFail($id);
        $req->update(['status' => 'checking_technical']);
        return redirect()->route('dashboard')->with('success', 'Disetujui Pudir 1. Menunggu pengecekan teknis.');
    }

    /**
     * TAHAP 3: Update Rekomendasi Teknis (Oleh Tim Pemelihara)
     */
    public function updateTechnical(Request $request, $id) 
    {
        $request->validate([
            'recommendation' => 'required|string',
            'type' => 'required|in:Internal,External',
            'cost' => 'required|numeric|min:0'
        ]);

        $req = MaintenanceRequest::where('status', 'checking_technical')->findOrFail($id);
        $req->update([
            'technical_recommendation' => $request->recommendation,
            'repair_type' => $request->type, 
            'estimated_cost' => $request->cost,
            'status' => 'pending_pudir2' 
        ]);
        
        return redirect()->route('dashboard')->with('success', 'Rekomendasi teknis diperbarui. Menunggu persetujuan anggaran (Pudir 2).');
    }

    /**
     * TAHAP 4: Persetujuan Pudir 2 (Anggaran/Keuangan)
     */
    public function approvePudir2($id) 
    {
        $req = MaintenanceRequest::where('status', 'pending_pudir2')->findOrFail($id);
        $req->update(['status' => 'repairing']); 
        
        return redirect()->route('dashboard')->with('success', 'Anggaran disetujui. Proses perbaikan dimulai.');
    }

    /**
     * TAHAP 5: Konfirmasi Pekerjaan Selesai
     */
    public function finishWork($id) 
    {
        $req = MaintenanceRequest::where('status', 'repairing')->findOrFail($id);
        $req->update(['status' => 'waiting_verification']);
        return redirect()->route('dashboard')->with('success', 'Pekerjaan selesai. Menunggu verifikasi admin unit.');
    }

    /**
     * TAHAP 6: Verifikasi Hasil Pekerjaan oleh Kepala Lab
     */
    public function verifyWork($id) 
    {
        $req = MaintenanceRequest::where('status', 'waiting_verification')->findOrFail($id);
        $req->update(['status' => 'ready_to_close']);
        return redirect()->route('dashboard')->with('success', 'Hasil perbaikan terverifikasi oleh unit kerja.');
    }

    /**
     * TAHAP 7: Penutupan Tiket (Closed)
     */
    public function closeTicket($id) 
    {
        $req = MaintenanceRequest::where('status', 'ready_to_close')->findOrFail($id);
        $req->update(['status' => 'closed']);
        
        return redirect()->route('dashboard')->with('success', 'Tiket ditutup. Proses pemeliharaan selesai.');
    }

    /**
     * MENGHAPUS PENGAJUAN (Hanya untuk pengajuan baru)
     */
    public function destroy($id)
    {
        $req = MaintenanceRequest::findOrFail($id);
        if ($req->status !== 'pending_kaprodi') {
            return back()->with('error', 'Hanya pengajuan awal yang bisa dihapus.');
        }
        
        return DB::transaction(function () use ($req) {
            if ($req->foto_kerusakan) {
                Storage::disk('public')->delete($req->foto_kerusakan);
            }
            $req->delete();
            return back()->with('success', 'Pengajuan berhasil dibatalkan.');
        });
    }

    /**
     * Cetak Bukti Perbaikan (Form BMN)
     */
    public function print($id)
    {
        $item = MaintenanceRequest::with(['barang.lab', 'user', 'lab'])->findOrFail($id);
        return view('admin.maintenance.print', compact('item'));
    }
}