<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Unit; 
use App\Models\Prodi; 
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{
    /**
     * Menampilkan daftar pengguna dengan fitur pencarian dan paginasi.
     */
    public function index(Request $request) 
    {
        $search = $request->input('search');

        $users = User::when($search, function ($query) use ($search) {
            $query->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('role', 'like', "%{$search}%");
        })
        ->latest()
        ->paginate(10);

        // Ambil data pendukung untuk dropdown di modal tambah/edit
        $laboratoriums = Unit::all(); 
        $prodis = Prodi::all(); 

        return view('admin.users.index', compact('users', 'laboratoriums', 'prodis'));
    }

    /**
     * Menyimpan pengguna baru ke database.
     */
    public function store(Request $request) 
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:6',
            'role' => 'required',
            'prodi_id' => 'nullable|integer',
            'lab_id' => 'nullable|integer'
        ]);

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            // Simpan prodi_id hanya jika role adalah Kaprodi
            'prodi_id' => $request->role === 'Kaprodi' ? $request->prodi_id : null,
            // Simpan lab_id hanya jika role adalah Kepala Lab
            'lab_id' => $request->role === 'Kepala Lab' ? $request->lab_id : null, 
        ]);

        return back()->with('success', 'User berhasil ditambahkan.');
    }

    /**
     * Memperbarui data pengguna.
     */
    public function update(Request $request, $id) 
    {
        $user = User::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $id,
            'role' => 'required',
            'prodi_id' => 'nullable|integer',
            'lab_id' => 'nullable|integer'
        ]);

        $updateData = $request->only(['name', 'email', 'role']);

        // Logika pembersihan dan pengaturan ID berdasarkan perubahan role
        if ($request->role === 'Kepala Lab') {
            $updateData['lab_id'] = $request->lab_id;
            $updateData['prodi_id'] = null;
        } elseif ($request->role === 'Kaprodi') {
            $updateData['prodi_id'] = $request->prodi_id;
            $updateData['lab_id'] = null;
        } else {
            // Role lain (Super Admin, Pudir, dll) tidak memiliki keterikatan Lab/Prodi spesifik
            $updateData['lab_id'] = null;
            $updateData['prodi_id'] = null;
        }

        $user->update($updateData);
        
        if ($request->filled('password')) {
            $user->update(['password' => Hash::make($request->password)]);
        }

        return back()->with('success', 'User berhasil diperbarui.');
    }

    /**
     * Menghapus pengguna (dengan proteksi diri sendiri).
     */
    public function destroy($id) 
    {
        $user = User::findOrFail($id);
        
        if ($user->is(Auth::user())) {
            return back()->with('error', 'Anda tidak dapat menghapus akun Anda sendiri.');
        }

        $user->delete();
        return back()->with('success', 'User berhasil dihapus.');
    }
}