<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class MaintenanceRequest extends Model
{
    use HasFactory;

    protected $table = 'maintenance_requests';

    /**
     * Mass Assignment
     * Pastikan barang_id sudah masuk untuk menggantikan input manual.
     */
    protected $fillable = [
        'user_id',
        'barang_id', // Foreign key baru ke Master Barang
        'id_lab', 
        'issue_description',
        'foto_kerusakan',
        'urgency',
        'damage_level',
        'request_date',
        'status',
        'technical_recommendation',
        'repair_type',
        'estimated_cost',
        'rejection_note'
    ];

    /**
     * Casting Data
     */
    protected $casts = [
        'request_date' => 'datetime',
        'estimated_cost' => 'decimal:2',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Relasi ke Model Barang (Master Barang)
     * Menghubungkan pengajuan dengan aset alat yang valid.
     */
    public function barang(): BelongsTo
    {
        return $this->belongsTo(Barang::class, 'barang_id');
    }

    /**
     * Relasi ke Model Laboratorium
     * Menampilkan lokasi fisik alat saat dilaporkan.
     */
    public function lab(): BelongsTo
    {
        return $this->belongsTo(Laboratorium::class, 'id_lab');
    }

    /**
     * Relasi ke Model User (Pelapor)
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Accessor: Status Label
     * Digunakan untuk warna badge di dashboard (Light Mode).
     */
    public function getStatusLabelAttribute()
    {
        return match($this->status) {
            'closed' => 'success',
            'repairing' => 'warning',
            'rejected' => 'danger',
            'pending_kaprodi', 'pending_pudir1', 'pending_pudir2' => 'primary',
            default => 'secondary'
        };
    }

    /**
     * Accessor: Ticket ID (Formatted)
     * Mengubah ID database menjadi format TIC-0001.
     */
    public function getFormattedIdAttribute()
    {
        return 'TIC-' . str_pad($this->id, 4, '0', STR_PAD_LEFT);
    }
}