<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Announcement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class AnnouncementController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Announcement::query();

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where('title', 'LIKE', "%{$search}%")
                  ->orWhere('content', 'LIKE', "%{$search}%");
        }

        // Filter by status
        if ($request->has('status') && $request->status !== '') {
            if ($request->status === 'published') {
                $query->where('is_published', true);
            } elseif ($request->status === 'draft') {
                $query->where('is_published', false);
            }
        }

        $announcements = $query->latest()->paginate(15);

        return view('admin.announcements.index', compact('announcements'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.announcements.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'attachment' => 'nullable|file|mimes:pdf,doc,docx|max:5120', // 5MB
            'is_published' => 'boolean',
            'published_at' => 'nullable|date|after:now',
        ]);

        $data = $request->only(['title', 'content', 'is_published', 'published_at']);
        $data['user_id'] = auth()->id();

        // Handle attachment upload
        if ($request->hasFile('attachment')) {
            $fileName = time() . '_' . $request->attachment->getClientOriginalName();
            $request->attachment->storeAs('public/announcements', $fileName);
            $data['attachment'] = 'announcements/' . $fileName;
        }

        // Set published_at if not provided and is_published is true
        if ($data['is_published'] && empty($data['published_at'])) {
            $data['published_at'] = now();
        }

        Announcement::create($data);

        return redirect()->route('admin.announcements.index')
                        ->with('success', 'Pengumuman berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Announcement $announcement)
    {
        return view('admin.announcements.show', compact('announcement'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Announcement $announcement)
    {
        return view('admin.announcements.edit', compact('announcement'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Announcement $announcement)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'attachment' => 'nullable|file|mimes:pdf,doc,docx|max:5120',
            'is_published' => 'boolean',
            'published_at' => 'nullable|date',
        ]);

        $data = $request->only(['title', 'content', 'is_published', 'published_at']);

        // Handle attachment upload
        if ($request->hasFile('attachment')) {
            // Delete old attachment if exists
            if ($announcement->attachment && Storage::exists('public/' . $announcement->attachment)) {
                Storage::delete('public/' . $announcement->attachment);
            }

            $fileName = time() . '_' . $request->attachment->getClientOriginalName();
            $request->attachment->storeAs('public/announcements', $fileName);
            $data['attachment'] = 'announcements/' . $fileName;
        }

        // Set published_at if not provided and is_published is true
        if ($data['is_published'] && empty($data['published_at'])) {
            $data['published_at'] = now();
        }

        $announcement->update($data);

        return redirect()->route('admin.announcements.index')
                        ->with('success', 'Pengumuman berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Announcement $announcement)
    {
        // Delete attachment if exists
        if ($announcement->attachment && Storage::exists('public/' . $announcement->attachment)) {
            Storage::delete('public/' . $announcement->attachment);
        }

        $announcement->delete();

        return redirect()->route('admin.announcements.index')
                        ->with('success', 'Pengumuman berhasil dihapus.');
    }
}
