<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Profile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ProfileController extends Controller
{
    private $sections = [
        'visi_misi' => 'Visi & Misi',
        'sejarah' => 'Sejarah',
        'dosen' => 'Dosen',
    ];

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Profile::query();

        // Filter by section
        if ($request->has('section') && $request->section !== '') {
            $query->where('section', $request->section);
        }

        // Filter by status
        if ($request->has('status') && $request->status !== '') {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        $profiles = $query->orderBy('section')->orderBy('order')->paginate(15);

        return view('admin.profiles.index', compact('profiles') + ['sections' => $this->sections]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.profiles.create', ['sections' => $this->sections]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'section' => 'required|string|in:visi_misi,sejarah,dosen',
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $data = $request->only(['section', 'title', 'content', 'order', 'is_active']);

        // Handle image upload
        if ($request->hasFile('image')) {
            $imageName = time() . '.' . $request->image->extension();
            $request->image->storeAs('public/profiles', $imageName);
            $data['image'] = 'profiles/' . $imageName;
        }

        Profile::create($data);

        return redirect()->route('admin.profiles.index')
                        ->with('success', 'Konten profil berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Profile $profile)
    {
        return view('admin.profiles.show', compact('profile') + ['sections' => $this->sections]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Profile $profile)
    {
        return view('admin.profiles.edit', compact('profile', 'sections'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Profile $profile)
    {
        $request->validate([
            'section' => 'required|string|in:visi_misi,sejarah,dosen',
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'order' => 'integer|min:0',
            'is_active' => 'boolean',
        ]);

        $data = $request->only(['section', 'title', 'content', 'order', 'is_active']);

        // Handle image upload
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($profile->image && Storage::exists('public/' . $profile->image)) {
                Storage::delete('public/' . $profile->image);
            }

            $imageName = time() . '.' . $request->image->extension();
            $request->image->storeAs('public/profiles', $imageName);
            $data['image'] = 'profiles/' . $imageName;
        }

        $profile->update($data);

        return redirect()->route('admin.profiles.index')
                        ->with('success', 'Konten profil berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Profile $profile)
    {
        // Delete image if exists
        if ($profile->image && Storage::exists('public/' . $profile->image)) {
            Storage::delete('public/' . $profile->image);
        }

        $profile->delete();

        return redirect()->route('admin.profiles.index')
                        ->with('success', 'Konten profil berhasil dihapus.');
    }
}
